<?php
/**
 * Plugin Name: WooCommerce Shipping Rounding Fixer
 * Description: Adds a "Cost (incl. VAT)" field to all WooCommerce shipping methods and splits it into net + tax at checkout.
 * Version:     1.1.1
 * Author:      Radu Ganea
 * Text Domain: woo-shipping-rounding-fixer
 * Domain Path: /languages
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Load plugin text domain from 'languages' directory.
 *
 * @return void
 */
function wsrf_load_plugin_textdomain() {
	load_plugin_textdomain( 'woo-shipping-rounding-fixer', false, dirname( plugin_basename( __FILE__ ) ) . '/languages' );
}
add_action( 'plugins_loaded', 'wsrf_load_plugin_textdomain' );

/**
 * Evaluate shipping cost formula.
 *
 * Supports [qty], [cost], and [fee percent="x" min_fee="y" max_fee="z"] tokens.
 *
 * @param string $formula The formula string entered by admin.
 * @param array  $package The WooCommerce shipping package.
 *
 * @return float Calculated cost.
 */
function wsrf_evaluate_formula( $formula, $package ) {
	$qty  = isset( $package['contents'] ) ? array_sum( wp_list_pluck( $package['contents'], 'quantity' ) ) : WC()->cart->get_cart_contents_count();
	$cost = isset( $package['contents_cost'] ) ? $package['contents_cost'] : WC()->cart->get_cart_contents_total();

	$expr = str_replace( [ '[qty]', '[cost]' ], [ $qty, $cost ], $formula );
	$expr = preg_replace_callback( '/\\[fee\\s+percent="([\\d\\.]+)"\\s+min_fee="([\\d\\.]+)"\\s+max_fee="([\\d\\.]*)"\\]/', function( $m ) use ( $cost ) {
		$pct = floatval( $m[1] ) / 100;
		$min = floatval( $m[2] );
		$max = $m[3] !== '' ? floatval( $m[3] ) : null;
		$fee = max( $cost * $pct, $min );
		if ( $max !== null ) {
			$fee = min( $fee, $max );
		}
		return $fee;
	}, $expr );

	if ( class_exists( 'WC_Eval_Math' ) ) {
		$value = WC_Eval_Math::evaluate( $expr );
	} else {
		$safe = preg_replace( '/[^\\d\\+\\-\\*\\/\\.\\(\\)]/', '', $expr );
		$value = eval( 'return ' . $safe . ';' );
	}

	return round( floatval( $value ), wc_get_price_decimals() );
}

/**
 * Register 'Cost (incl. VAT)' field for all available shipping methods.
 *
 * @return void
 */
function wsrf_register_shipping_field_hooks() {
	if ( ! class_exists( 'WC_Shipping_Zones' ) ) {
		return;
	}
	foreach ( WC()->shipping()->get_shipping_methods() as $id => $method ) {
		add_filter( "woocommerce_shipping_instance_form_fields_{$id}", 'wsrf_add_cost_incl_vat_field', 20 );
		add_filter( "woocommerce_{$id}_instance_form_fields", 'wsrf_add_cost_incl_vat_field', 20 );
	}
}
add_action( 'init', 'wsrf_register_shipping_field_hooks', 20 );

/**
 * Add 'Cost (incl. VAT)' field to shipping method settings.
 *
 * Note: This filter passes only one parameter — $fields.
 *
 * @param array $fields Existing fields.
 *
 * @return array Modified fields.
 */
function wsrf_add_cost_incl_vat_field( $fields ) {
	$fields['cost_incl_vat'] = [
		'title'       => __( 'Cost (incl. VAT)', 'woo-shipping-rounding-fixer' ),
		'type'        => 'text',
		'description' => __( 'Enter a fixed or formula-based cost including VAT. 10.00 or [qty]*2 + [fee percent="5" min_fee="2" max_fee="5"]', 'woo-shipping-rounding-fixer' ),
		'default'     => '',
		'placeholder' => '',
		'css'         => 'width:100%;',
		'desc_tip'    => false,
	];
	return $fields;
}


add_filter( 'woocommerce_package_rates', 'wsrf_adjust_shipping_rate_costs', 10, 2 );

/**
 * Adjust shipping cost and tax based on admin-defined gross (incl. VAT).
 *
 * @param array $rates   Shipping rates.
 * @param array $package Shipping package data.
 *
 * @return array Modified rates.
 */
function wsrf_adjust_shipping_rate_costs( $rates, $package ) {
	foreach($rates as $key => $rate){
        $method    = WC_Shipping_Zones::get_shipping_method($rate->instance_id);
        $gross_raw = $method->get_option('cost_incl_vat');
        $gross_raw = str_replace(",", ".", $gross_raw);
        if(''===$gross_raw) continue;
        $net_raw   = $method->get_option('cost');

        $gross = is_numeric($gross_raw) ? floatval($gross_raw) : wsrf_evaluate_formula($gross_raw, $package);

        $net   = is_numeric($net_raw) ? floatval($net_raw) : wsrf_evaluate_formula($net_raw, $package);
        $tax   = round($gross - $net, wc_get_price_decimals());

        $rate->set_cost( wc_format_decimal($net, wc_get_price_decimals()) );
        $txs = [];
        foreach( WC_Tax::get_shipping_tax_rates() as $tx_id => $tx ) {
            $txs[$tx_id] = $tax;
        }
        $rate->set_taxes($txs);
        $rates[$key] = $rate;
    }
    return $rates;
}

/**
 * Save gross shipping cost to order item meta when creating shipping item.
 *
 * @param WC_Order_Item_Shipping $item        Shipping item object.
 * @param int                    $package_key  Index of the shipping package.
 * @param array                  $package      Package details.
 * @param WC_Order               $order        Order object.
 *
 * @return void
 */
function wsrf_store_shipping_gross_on_item( $item, $package_key, $package, $order ) {
	$session_rates = WC()->session->get( 'shipping_for_package_' . $package_key );
    $rates = is_array( $session_rates ) && isset( $session_rates['rates'] ) ? $session_rates['rates'] : [];

	$method_id = $item->get_method_id() . ':' . $item->get_instance_id();

	if ( isset( $rates[ $method_id ] ) && isset( $rates[ $method_id ]->wsrf_cost_incl_vat ) ) {
		$item->add_meta_data( '_wsrf_gross_shipping_cost', $rates[ $method_id ]->wsrf_cost_incl_vat, true );
	}
}
add_action( 'woocommerce_checkout_create_order_shipping_item', 'wsrf_store_shipping_gross_on_item', 10, 4 );

/**
 * Correct shipping tax after order creation (for HPOS compatibility).
 *
 * @param int $order_id WooCommerce Order ID.
 *
 * @return void
 */
function wsrf_correct_shipping_tax_after_order( $order_id ) {
	$order = wc_get_order( $order_id );
	if ( ! $order ) {
		return;
	}
	$net = (float) $order->get_shipping_total();
	$gross = 0.0;

	foreach ( $order->get_items( 'shipping' ) as $item ) {
		$g = $item->get_meta( '_wsrf_gross_shipping_cost', true );
		$gross += floatval( $g );
	}

	$tax = round( $gross - $net, wc_get_price_decimals() );
	$order->set_shipping_tax( wc_format_decimal( $tax, wc_get_price_decimals() ) );
	$order->save();
}
add_action( 'woocommerce_new_order', 'wsrf_correct_shipping_tax_after_order', 20, 1 );

/**
 * Ensure order shipping total getter reflects gross cost.
 *
 * @param float     $value  Original total.
 * @param WC_Order  $order  Order object.
 *
 * @return float Filtered total.
 */
function wsrf_override_shipping_total_getter( $value, $order ) {
	$total = 0;
	foreach ( $order->get_items( 'shipping' ) as $item ) {
		$gross = $item->get_meta( '_wsrf_gross_shipping_cost', true );
		if ( $gross ) {
			$total += floatval( $gross );
		}
	}
	return $total > 0 ? wc_format_decimal( $total ) : $value;
}
add_filter( 'woocommerce_order_get_shipping_total', 'wsrf_override_shipping_total_getter', 20, 2 );

/**
 * Ensure order shipping tax getter reflects (gross - net).
 *
 * @param float     $value  Original tax.
 * @param WC_Order  $order  Order object.
 *
 * @return float Filtered tax.
 */
function wsrf_override_shipping_tax_getter( $value, $order ) {
	$sum = 0;
	foreach ( $order->get_items( 'shipping' ) as $item ) {
		$net   = (float) $item->get_total();
		$gross = (float) $item->get_meta( '_wsrf_gross_shipping_cost', true );
		if ( ! $gross ) {
			$gross = $net + array_sum( array_map( 'floatval', (array) $item->get_taxes()['total'] ) );
		}
		$sum += round( $gross - $net, wc_get_price_decimals() );
	}
	return round( $sum, wc_get_price_decimals() );
}
add_filter( 'woocommerce_order_get_shipping_tax', 'wsrf_override_shipping_tax_getter', 20, 2 );
